/*
 * This file is part of Araknemu.
 *
 * Araknemu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Araknemu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Araknemu.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2017-2019 Vincent Quatrevieux
 */

package fr.quatrevieux.araknemu.game.fight.builder;

import fr.arakne.utils.value.helper.RandomUtil;
import fr.quatrevieux.araknemu.game.GameBaseCase;
import fr.quatrevieux.araknemu.game.exploration.map.ExplorationMapService;
import fr.quatrevieux.araknemu.game.fight.Fight;
import fr.quatrevieux.araknemu.game.fight.FightService;
import fr.quatrevieux.araknemu.game.fight.fighter.Fighter;
import fr.quatrevieux.araknemu.game.fight.fighter.player.PlayerFighter;
import fr.quatrevieux.araknemu.game.fight.state.PlacementState;
import fr.quatrevieux.araknemu.game.fight.team.SimpleTeam;
import fr.quatrevieux.araknemu.game.fight.turn.order.AlternateTeamFighterOrder;
import fr.quatrevieux.araknemu.game.fight.type.FightType;
import fr.quatrevieux.araknemu.game.player.GamePlayer;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import java.util.ArrayList;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotEquals;
import static org.junit.jupiter.api.Assertions.assertSame;
import static org.junit.jupiter.api.Assertions.assertThrows;

class BaseBuilderTest extends GameBaseCase {
    private BaseBuilder builder;
    private FightType type;

    @Override
    @BeforeEach
    public void setUp() throws Exception {
        super.setUp();

        dataSet.pushMaps().pushSubAreas().pushAreas();

        type = Mockito.mock(FightType.class);
        builder = new BaseBuilder(container.get(FightService.class), new RandomUtil(), type);
    }

    @Test
    void buildSimple() throws Exception {
        PlayerFighter fighter = new PlayerFighter(gamePlayer());
        PlayerFighter other = new PlayerFighter(makeOtherPlayer());

        builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, fighter, startPlaces, number));
        builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, other, startPlaces, number));
        builder.map(container.get(ExplorationMapService.class).load(10340));

        Fight fight = builder.build(1);

        // Call join fight on fighters
        new PlacementState().start(fight);

        fight.start(new AlternateTeamFighterOrder());

        assertSame(type, fight.type());
        assertCount(2, fight.teams());
        assertCount(1, fight.team(0).fighters());
        assertCount(1, fight.team(1).fighters());
        assertContainsOnly(SimpleTeam.class, fight.teams());
        assertContainsOnly(PlayerFighter.class, fight.fighters().all());
        assertEquals(1, fight.id());
    }

    @Test
    void buildWithoutTeam() {
        builder.map(container.get(ExplorationMapService.class).load(10340));

        assertThrows(IllegalStateException.class, () -> builder.build(1));
    }

    @Test
    void buildWithoutMap() {
        assertThrows(IllegalStateException.class, () -> builder.build(1));
    }

    @Test
    void buildWillRandomizeTeamNumbers() throws Exception {
        PlayerFighter fighter = new PlayerFighter(gamePlayer());
        PlayerFighter other = new PlayerFighter(makeOtherPlayer());

        builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, fighter, startPlaces, number));
        builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, other, startPlaces, number));
        builder.map(container.get(ExplorationMapService.class).load(10340));

        int nbTeam0 = 0;

        for (int i = 0; i < 100; ++i) {
            Fight fight = builder.build(1);

            assertSame(type, fight.type());
            assertCount(2, fight.teams());
            assertCount(1, fight.team(0).fighters());
            assertCount(1, fight.team(1).fighters());
            assertEquals(1, fight.id());

            Fighter firstFighter = new ArrayList<>(fight.team(0).fighters()).get(0);

            if (firstFighter.id() == gamePlayer().id()) {
                ++nbTeam0;
            }
        }

        assertNotEquals(0, nbTeam0);
        assertNotEquals(100, nbTeam0);
    }

    @Test
    void buildWithoutRandomizeTeam() throws Exception {
        GamePlayer player1 = gamePlayer();
        GamePlayer player2 = makeOtherPlayer();

        for (int i = 0; i < 100; ++i) {
            builder = new BaseBuilder(container.get(FightService.class), null, type);

            PlayerFighter fighter = new PlayerFighter(player1);
            PlayerFighter other = new PlayerFighter(player2);

            builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, fighter, startPlaces, number));
            builder.addTeam((fight, number, startPlaces) -> new SimpleTeam(fight, other, startPlaces, number));
            builder.map(container.get(ExplorationMapService.class).load(10340));

            Fight fight = builder.build(1);

            // Call join fight on fighters
            new PlacementState().start(fight);

            fight.start(new AlternateTeamFighterOrder());

            assertSame(type, fight.type());
            assertCount(2, fight.teams());
            assertCount(1, fight.team(0).fighters());
            assertCount(1, fight.team(1).fighters());
            assertEquals(1, fight.id());

            Fighter firstFighter = new ArrayList<>(fight.team(0).fighters()).get(0);

            assertEquals(gamePlayer().id(), firstFighter.id());
        }
    }
}
