/*
 * This file is part of Araknemu.
 *
 * Araknemu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Araknemu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Araknemu.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2017-2019 Vincent Quatrevieux
 */

package fr.quatrevieux.araknemu.game.fight.ai.simulation.effect;

import fr.quatrevieux.araknemu.data.value.EffectArea;
import fr.quatrevieux.araknemu.game.fight.Fight;
import fr.quatrevieux.araknemu.game.fight.FightBaseCase;
import fr.quatrevieux.araknemu.game.fight.ai.FighterAI;
import fr.quatrevieux.araknemu.game.fight.ai.action.logic.NullGenerator;
import fr.quatrevieux.araknemu.game.fight.ai.simulation.CastSimulation;
import fr.quatrevieux.araknemu.game.fight.ai.simulation.SpellScore;
import fr.quatrevieux.araknemu.game.fight.castable.CastScope;
import fr.quatrevieux.araknemu.game.fight.fighter.Fighter;
import fr.quatrevieux.araknemu.game.fight.fighter.player.PlayerFighter;
import fr.quatrevieux.araknemu.game.fight.map.FightCell;
import fr.quatrevieux.araknemu.game.spell.Spell;
import fr.quatrevieux.araknemu.game.spell.SpellConstraints;
import fr.quatrevieux.araknemu.game.spell.effect.SpellEffect;
import fr.quatrevieux.araknemu.game.spell.effect.area.CellArea;
import fr.quatrevieux.araknemu.game.spell.effect.area.CircleArea;
import fr.quatrevieux.araknemu.game.spell.effect.target.SpellEffectTarget;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import static org.junit.jupiter.api.Assertions.assertEquals;

class StealCharacteristicSimulatorTest extends FightBaseCase {
    private Fight fight;
    private PlayerFighter fighter;
    private PlayerFighter target;
    private FighterAI ai;

    @Override
    @BeforeEach
    public void setUp() throws Exception {
        super.setUp();

        fight = createFight();
        fighter = player.fighter();
        target = other.fighter();
        ai = new FighterAI(fighter, fight, new NullGenerator());
    }

    @Test
    void simulateSimple() {
        StealCharacteristicSimulator simulator = new StealCharacteristicSimulator();

        SpellEffect effect = Mockito.mock(SpellEffect.class);
        Spell spell = Mockito.mock(Spell.class);
        SpellConstraints constraints = Mockito.mock(SpellConstraints.class);

        Mockito.when(effect.min()).thenReturn(10);
        Mockito.when(effect.area()).thenReturn(new CellArea());
        Mockito.when(effect.target()).thenReturn(SpellEffectTarget.DEFAULT);
        Mockito.when(spell.constraints()).thenReturn(constraints);
        Mockito.when(constraints.freeCell()).thenReturn(false);

        CastSimulation simulation = new CastSimulation(spell, fighter, target.cell());

        CastScope<Fighter, FightCell> scope = makeCastScope(fighter, spell, effect, target.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));

        assertEquals(10, simulation.selfBoost());
        assertEquals(-10, simulation.enemiesBoost());
    }

    @Test
    void shouldIgnoreSelfTarget() {
        StealCharacteristicSimulator simulator = new StealCharacteristicSimulator();

        SpellEffect effect = Mockito.mock(SpellEffect.class);
        Spell spell = Mockito.mock(Spell.class);
        SpellConstraints constraints = Mockito.mock(SpellConstraints.class);

        Mockito.when(effect.min()).thenReturn(10);
        Mockito.when(effect.area()).thenReturn(new CellArea());
        Mockito.when(effect.target()).thenReturn(SpellEffectTarget.DEFAULT);
        Mockito.when(spell.constraints()).thenReturn(constraints);
        Mockito.when(constraints.freeCell()).thenReturn(false);

        CastSimulation simulation = new CastSimulation(spell, fighter, fighter.cell());

        CastScope<Fighter, FightCell> scope = makeCastScope(fighter, spell, effect, fighter.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));

        assertEquals(0, simulation.selfBoost());
        assertEquals(0, simulation.enemiesBoost());
    }

    @Test
    void simulateBuff() {
        StealCharacteristicSimulator simulator = new StealCharacteristicSimulator();

        SpellEffect effect = Mockito.mock(SpellEffect.class);
        Spell spell = Mockito.mock(Spell.class);
        SpellConstraints constraints = Mockito.mock(SpellConstraints.class);

        Mockito.when(effect.min()).thenReturn(10);
        Mockito.when(effect.area()).thenReturn(new CellArea());
        Mockito.when(effect.target()).thenReturn(SpellEffectTarget.DEFAULT);
        Mockito.when(effect.duration()).thenReturn(2);
        Mockito.when(spell.constraints()).thenReturn(constraints);
        Mockito.when(constraints.freeCell()).thenReturn(false);

        CastSimulation simulation = new CastSimulation(spell, fighter, target.cell());

        CastScope<Fighter, FightCell> scope = makeCastScope(fighter, spell, effect, target.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));

        assertEquals(20, simulation.selfBoost());
        assertEquals(-20, simulation.enemiesBoost());

        Mockito.when(effect.duration()).thenReturn(5);
        simulation = new CastSimulation(spell, fighter, target.cell());
        scope = makeCastScope(fighter, spell, effect, target.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));
        assertEquals(50, simulation.selfBoost());
        assertEquals(-50, simulation.enemiesBoost());

        Mockito.when(effect.duration()).thenReturn(20);
        simulation = new CastSimulation(spell, fighter, target.cell());
        scope = makeCastScope(fighter, spell, effect, target.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));
        assertEquals(100, simulation.selfBoost());
        assertEquals(-100, simulation.enemiesBoost());

        Mockito.when(effect.duration()).thenReturn(-1);
        simulation = new CastSimulation(spell, fighter, target.cell());
        scope = makeCastScope(fighter, spell, effect, target.cell());
        simulator.simulate(simulation, ai, scope.effects().get(0));
        assertEquals(100, simulation.selfBoost());
        assertEquals(-100, simulation.enemiesBoost());
    }

    @Test
    void simulateArea() {
        Fight fight = fightBuilder()
            .addSelf(fb -> fb.cell(220))
            .addEnemy(fb -> fb.player(other).cell(124))
            .addEnemy(fb -> fb.cell(125))
            .build(true)
        ;
        fight.nextState();

        fighter = player.fighter();

        ai = new FighterAI(fighter, fight, new NullGenerator());

        StealCharacteristicSimulator simulator = new StealCharacteristicSimulator();

        SpellEffect effect = Mockito.mock(SpellEffect.class);
        Spell spell = Mockito.mock(Spell.class);
        SpellConstraints constraints = Mockito.mock(SpellConstraints.class);

        Mockito.when(effect.min()).thenReturn(10);
        Mockito.when(effect.area()).thenReturn(new CircleArea(new EffectArea(EffectArea.Type.CIRCLE, 10)));
        Mockito.when(effect.target()).thenReturn(SpellEffectTarget.DEFAULT);
        Mockito.when(spell.constraints()).thenReturn(constraints);
        Mockito.when(constraints.freeCell()).thenReturn(false);

        CastSimulation simulation = new CastSimulation(spell, fighter, fight.map().get(124));

        CastScope<Fighter, FightCell> scope = makeCastScope(fighter, spell, effect, fight.map().get(124));
        simulator.simulate(simulation, ai, scope.effects().get(0));

        assertEquals(20, simulation.selfBoost());
        assertEquals(-20, simulation.enemiesBoost());
    }

    @Test
    void scorePositive() {
        StealCharacteristicSimulator simulator = new StealCharacteristicSimulator();

        SpellEffect effect = Mockito.mock(SpellEffect.class);
        Spell spell = Mockito.mock(Spell.class);
        SpellConstraints constraints = Mockito.mock(SpellConstraints.class);

        Mockito.when(effect.min()).thenReturn(2);
        Mockito.when(effect.max()).thenReturn(3);
        Mockito.when(effect.area()).thenReturn(new CellArea());
        Mockito.when(effect.target()).thenReturn(SpellEffectTarget.DEFAULT);
        Mockito.when(spell.constraints()).thenReturn(constraints);
        Mockito.when(constraints.freeCell()).thenReturn(false);

        SpellScore score = new SpellScore();
        simulator.score(score, effect, fighter.characteristics());

        assertEquals(2.0, score.score());
        assertEquals(2.0, score.debuff());
    }
}
