/*
 * This file is part of Araknemu.
 *
 * Araknemu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Araknemu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Araknemu.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2017-2019 Vincent Quatrevieux
 */

package fr.quatrevieux.araknemu.game.fight.turn.order;

import fr.quatrevieux.araknemu.game.fight.fighter.Fighter;
import fr.quatrevieux.araknemu.game.fight.fighter.PlayableFighter;
import fr.quatrevieux.araknemu.game.fight.team.FightTeam;
import fr.quatrevieux.araknemu.util.Asserter;
import org.checkerframework.checker.nullness.util.NullnessUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;

/**
 * Sort fighters by initiative with alternate teams
 */
public final class AlternateTeamFighterOrder implements FighterOrderStrategy {
    @Override
    public List<PlayableFighter> compute(List<FightTeam> teams) {
        final List<Queue<PlayableFighter>> fightersByTeam = computeTeamsOrder(teams);
        final List<PlayableFighter> orderedFighters = new ArrayList<>();

        boolean hasChanges;

        do {
            hasChanges = false;

            // Poll the next fighter on each teams
            for (Queue<PlayableFighter> fighters : fightersByTeam) {
                if (!fighters.isEmpty()) {
                    orderedFighters.add(fighters.remove());
                    hasChanges = true;
                }
            }
        } while (hasChanges);

        return orderedFighters;
    }

    private List<Queue<PlayableFighter>> computeTeamsOrder(List<FightTeam> teams) {
        final List<Queue<PlayableFighter>> fightersByTeam = new ArrayList<>();

        for (FightTeam team : teams) {
            // Sort team fighters by their initiative desc
            final Queue<PlayableFighter> fighters = new PriorityQueue<>(
                Asserter.castPositive(team.fighters().size()),
                (f1, f2) -> f2.characteristics().initiative() - f1.characteristics().initiative()
            );

            for (Fighter fighter : team.fighters()) {
                if (fighter instanceof PlayableFighter) {
                    fighters.add((PlayableFighter) fighter);
                }
            }

            fightersByTeam.add(fighters);
        }

        // Sort team by their best fighter initiative desc
        fightersByTeam.sort((t1, t2) -> NullnessUtil.castNonNull(t2.peek()).characteristics().initiative() - NullnessUtil.castNonNull(t1.peek()).characteristics().initiative());

        return fightersByTeam;
    }
}
