/*
 * This file is part of Araknemu.
 *
 * Araknemu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Araknemu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Araknemu.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2017-2024 Vincent Quatrevieux
 */

package fr.quatrevieux.araknemu.game.fight.castable.effect.buff;

import fr.quatrevieux.araknemu.data.constant.Characteristic;
import fr.quatrevieux.araknemu.game.fight.castable.FightCastScope;
import fr.quatrevieux.araknemu.game.fight.castable.effect.EffectValue;
import fr.quatrevieux.araknemu.game.fight.castable.effect.Element;
import fr.quatrevieux.araknemu.game.fight.castable.effect.handler.damage.Damage;
import fr.quatrevieux.araknemu.game.fight.castable.effect.handler.damage.ReflectedDamage;
import fr.quatrevieux.araknemu.game.fight.fighter.Fighter;
import fr.quatrevieux.araknemu.game.fight.turn.Turn;
import org.checkerframework.checker.index.qual.NonNegative;
import org.checkerframework.checker.index.qual.Positive;

/**
 * All hooks methods applied on fighter buff list
 */
public interface BuffListHooks {
    /**
     * @see BuffHook#onCast(FightBuff, FightCastScope)
     */
    public void onCast(FightCastScope cast);

    /**
     * Apply buffs when the fighter is a target of a cast
     *
     * If false is returned, the scope targets should be reloaded to call this hook on new targets,
     * and following hooks will be ignored.
     *
     * @return true to continue, or false if the target has changed (or is removed)
     *
     * @see BuffHook#onCastTarget(FightBuff, FightCastScope)
     */
    public boolean onCastTarget(FightCastScope cast);

    /**
     * @see BuffHook#onDirectDamage(FightBuff, Fighter, Damage)
     */
    public void onDirectDamage(Fighter caster, Damage value);

    /**
     * @see BuffHook#onIndirectDamage(FightBuff, Fighter, Damage)
     */
    public void onIndirectDamage(Fighter caster, Damage value);

    /**
     * @see BuffHook#onBuffDamage(FightBuff, FightBuff, Damage)
     */
    public void onBuffDamage(FightBuff poison, Damage value);

    /**
     * @see BuffHook#onDirectDamageApplied(FightBuff, Fighter, int)
     */
    public void onDirectDamageApplied(Fighter caster, @Positive int value);

    /**
     * @see BuffHook#onElementDamageApplied(FightBuff, Element element, int)
     */
    public void onElementDamageApplied(Element element, @NonNegative int actualDamage);

    /**
     * @see BuffHook#onDamageApplied(FightBuff, int)
     */
    public void onDamageApplied(@NonNegative int value);

    /**
     * @see BuffHook#onHealApplied(FightBuff, int)
     */
    public void onHealApplied(@NonNegative int value);

    /**
     * @see BuffHook#onReflectedDamage(FightBuff, ReflectedDamage)
     */
    public void onReflectedDamage(ReflectedDamage damage);

    /**
     * @see BuffHook#onStartTurn(FightBuff)
     */
    public boolean onStartTurn();

    /**
     * @see BuffHook#onEndTurn(FightBuff, Turn)
     */
    public void onEndTurn(Turn turn);

    /**
     * @see BuffHook#onCastDamage(FightBuff, Damage, Fighter)
     */
    public void onCastDamage(Damage damage, Fighter target);

    /**
     * @see BuffHook#onEffectValueCast(FightBuff, EffectValue)
     */
    public void onEffectValueCast(EffectValue value);

    /**
     * @see BuffHook#onEffectValueTarget(FightBuff, EffectValue)
     */
    public void onEffectValueTarget(EffectValue value);

    /**
     * @see BuffHook#onCharacteristicAltered(FightBuff, Characteristic, int)
     */
    public void onCharacteristicAltered(Characteristic characteristic, int value);
}
