/*
 * This file is part of Araknemu.
 *
 * Araknemu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Araknemu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Araknemu.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2017-2020 Vincent Quatrevieux
 */

package fr.quatrevieux.araknemu.data.living.entity.account;

import java.time.Instant;

/**
 * Store banishment account data
 * The banishment is linked to an account, a time period (start and end date)
 *
 * This entity is immutable
 */
public final class Banishment {
    private final int id;
    private final int accountId;
    private final Instant startDate;
    private final Instant endDate;
    private final String cause;
    private final int banisherId;

    public Banishment(int id, int accountId, Instant startDate, Instant endDate, String cause, int banisherId) {
        this.id = id;
        this.accountId = accountId;
        this.startDate = startDate;
        this.endDate = endDate;
        this.cause = cause;
        this.banisherId = banisherId;
    }

    public Banishment(int accountId, Instant startDate, Instant endDate, String cause, int banisherId) {
        this(-1, accountId, startDate, endDate, cause, banisherId);
    }

    /**
     * Get the banishment id (primary key)
     * This id is autogenerated
     */
    public int id() {
        return id;
    }

    /**
     * Get the banned account id
     * @see Account#id()
     */
    public int accountId() {
        return accountId;
    }

    /**
     * Get the banishment period start date
     */
    public Instant startDate() {
        return startDate;
    }

    /**
     * Get the banishment period end date
     */
    public Instant endDate() {
        return endDate;
    }

    /**
     * Get the banishment cause
     * The cause is a human readable string.
     * A format may be applied for automated banishment, but it's not normalized.
     */
    public String cause() {
        return cause;
    }

    /**
     * Get the game master account id who create this banishment
     *
     * @see Account#id()
     */
    public int banisherId() {
        return banisherId;
    }

    /**
     * Return a new instance with the banishment id
     */
    public Banishment withId(int id) {
        return new Banishment(id, accountId, startDate, endDate, cause, banisherId);
    }
}
