const fileInput = document.getElementById('file-input');
const dropZone = document.getElementById('drop-zone');
const previewSection = document.getElementById('preview-section');
const canvasContainer = document.querySelector('.canvas-container');
const closePreviewBtn = document.getElementById('close-preview');
const printBtn = document.getElementById('print-btn');

// Drag and drop handlers
dropZone.addEventListener('dragover', (e) => {
    e.preventDefault();
    dropZone.style.borderColor = 'var(--primary-color)';
    dropZone.style.background = '#eff6ff';
});

dropZone.addEventListener('dragleave', (e) => {
    e.preventDefault();
    dropZone.style.borderColor = 'var(--border-color)';
    dropZone.style.background = '#f8fafc';
});

dropZone.addEventListener('drop', (e) => {
    e.preventDefault();
    dropZone.style.borderColor = 'var(--border-color)';
    dropZone.style.background = '#f8fafc';

    if (e.dataTransfer.files.length > 0) {
        handleFile(e.dataTransfer.files[0]);
    }
});

fileInput.addEventListener('change', (e) => {
    if (e.target.files.length > 0) {
        handleFile(e.target.files[0]);
    }
});

closePreviewBtn.addEventListener('click', () => {
    previewSection.classList.add('hidden');
    dropZone.classList.remove('hidden');
    fileInput.value = '';
    canvasContainer.innerHTML = ''; // Clear canvases
});

let selectedDeviceAddress = null;

const configBtn = document.getElementById('config-btn');
const deviceModal = document.getElementById('device-modal');
const closeModalBtn = document.getElementById('close-modal');
const deviceList = document.getElementById('device-list');

// Config button handler
if (configBtn) {
    configBtn.addEventListener('click', async () => {
        deviceModal.classList.remove('hidden');
        await scanDevices();
    });
}

closeModalBtn.addEventListener('click', () => {
    deviceModal.classList.add('hidden');
});

async function scanDevices() {
    deviceList.innerHTML = '<li>Buscando dispositivos...</li>';

    const isCapacitor = !!window.Capacitor;
    if (!isCapacitor) {
        deviceList.innerHTML = '<li>Solo disponible en la App Android</li>';
        return;
    }

    try {
        const PrintPlugin = window.Capacitor.Plugins.Print;

        // Check/Request permissions
        const permStatus = await PrintPlugin.checkPermissions();
        if (permStatus.bluetooth !== 'granted') {
            const newStatus = await PrintPlugin.requestPermissions({ permissions: ['bluetooth'] });
            if (newStatus.bluetooth !== 'granted') {
                deviceList.innerHTML = '<li>Permisos de Bluetooth denegados</li>';
                return;
            }
        }

        const result = await PrintPlugin.scan();

        deviceList.innerHTML = '';

        if (result.devices && result.devices.length > 0) {
            result.devices.forEach(device => {
                const li = document.createElement('li');
                li.textContent = `${device.name || 'Desconocido'} (${device.address})`;
                if (device.address === selectedDeviceAddress) {
                    li.classList.add('selected');
                }

                li.addEventListener('click', () => {
                    selectedDeviceAddress = device.address;
                    // Update UI
                    document.querySelectorAll('#device-list li').forEach(el => el.classList.remove('selected'));
                    li.classList.add('selected');

                    // Save preference
                    localStorage.setItem('selectedPrinter', device.address);
                });

                deviceList.appendChild(li);
            });
        } else {
            deviceList.innerHTML = '<li>No se encontraron dispositivos vinculados</li>';
        }
    } catch (e) {
        console.error("Error scanning:", e);
        deviceList.innerHTML = `<li>Error: ${e.message}</li>`;
    }
}

// Load saved printer
const savedPrinter = localStorage.getItem('selectedPrinter');
if (savedPrinter) {
    selectedDeviceAddress = savedPrinter;
}

printBtn.addEventListener('click', async () => {
    // Check if running in Capacitor environment (Android/iOS)
    const isCapacitor = !!window.Capacitor;
    const platform = isCapacitor ? window.Capacitor.getPlatform() : 'web';
    const isNative = isCapacitor && platform !== 'web';

    if (isNative) {
        const PrintPlugin = window.Capacitor.Plugins.Print;

        if (!PrintPlugin) {
            window.print();
            return;
        }

        try {
            if (selectedDeviceAddress) {
                // Print to Bluetooth
                // Get the first canvas (assuming single page or image for now)
                const canvas = document.querySelector('canvas');
                if (!canvas) {
                    alert('No hay nada para imprimir');
                    return;
                }

                const base64Data = canvas.toDataURL('image/png').split(',')[1];

                await PrintPlugin.print({
                    type: 'bluetooth',
                    address: selectedDeviceAddress,
                    content: base64Data
                });

                alert('Impresión enviada');
            } else {
                // System print
                await PrintPlugin.print({ type: 'system' });
            }
        } catch (e) {
            console.error("Error printing native:", e);
            alert("Error al intentar imprimir: " + e.message);
        }
    } else {
        // Web browser
        window.print();
    }
});

async function handleFile(file) {
    // Clear previous content
    canvasContainer.innerHTML = '';

    if (file.type === 'application/pdf') {
        try {
            // Show preview, hide upload
            dropZone.classList.add('hidden');
            previewSection.classList.remove('hidden');

            const arrayBuffer = await file.arrayBuffer();
            const pdf = await pdfjsLib.getDocument(arrayBuffer).promise;

            // Render all pages
            for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
                await renderPage(pdf, pageNum);
            }
        } catch (error) {
            console.error('Error al cargar el PDF:', error);
            alert('Hubo un error al procesar el PDF.');
            resetUI();
        }
    } else if (file.type.startsWith('image/')) {
        try {
            // Show preview, hide upload
            dropZone.classList.add('hidden');
            previewSection.classList.remove('hidden');

            await renderImage(file);
        } catch (error) {
            console.error('Error al cargar la imagen:', error);
            alert('Hubo un error al procesar la imagen.');
            resetUI();
        }
    } else {
        alert('Por favor, sube un archivo válido (PDF, JPG, PNG).');
    }
}

function resetUI() {
    previewSection.classList.add('hidden');
    dropZone.classList.remove('hidden');
    fileInput.value = '';
    canvasContainer.innerHTML = '';
}

async function renderImage(file) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = (event) => {
            const img = new Image();
            img.onload = () => {
                const canvas = document.createElement('canvas');
                const context = canvas.getContext('2d');

                // Set canvas dimensions to image dimensions
                canvas.width = img.width;
                canvas.height = img.height;

                // Draw image
                context.drawImage(img, 0, 0);

                // Add to container
                canvasContainer.appendChild(canvas);
                resolve();
            };
            img.onerror = reject;
            img.src = event.target.result;
        };
        reader.onerror = reject;
        reader.readAsDataURL(file);
    });
}

async function renderPage(pdf, pageNum) {
    const page = await pdf.getPage(pageNum);

    // Calculate scale to fit width nicely or use a high quality scale
    // For thermal printers, we want high contrast and good resolution.
    // We'll render at scale 2 for quality, CSS will handle display size.
    const scale = 2.0;
    const viewport = page.getViewport({ scale: scale });

    const canvas = document.createElement('canvas');
    const context = canvas.getContext('2d');

    canvas.height = viewport.height;
    canvas.width = viewport.width;

    // Add to container
    canvasContainer.appendChild(canvas);

    const renderContext = {
        canvasContext: context,
        viewport: viewport
    };

    await page.render(renderContext).promise;
}
